# -*- coding: utf-8 -*-

""" zeusteach.py : Zeus teaching data を扱う  python library Module
Created on Fri Aug 14 16:49:50 2020
@author: c.tanaka


Todo:
   TODOリストを記載
    * エラー処理を正しく実装する
    *
"""
import inspect
import linecache
import copy
import os
import json

#======================================================================
class ZeusTeach():
    """ ZeusTeach Class の説明
    　User Python Applicationで、Zeus teaching data を使用する場合に使用するClass
    　Teaching dataを使用する（宣言する）場合に、ZeusTeachクラスのインスタンスを生成し、
    　TPosition()関数を使って、Position型のTeaching変数や配列を作成する。
    　Joint型のTeaching変数や配列は、TJoint()関数を使って作成する。
        1. TPosition():     Position型の teaching data 変数や配列を定義
        2. TJoint():        Joint型の teaching data 変数や配列を定義
        
    Examples:
        zt = ZeusTeach()                                # ZeusTeachクラス
        pickup1     = zt.TPosition()                    # Position型の teaching data を定義
        pos_list    = zt.TPosition(5)                   # Position型でサイズ=5の teaching data 配列（リスト）を定義
        joint_list  = zt.TJoint(2)                      # Joint型でサイズ=2の teaching data 配列（リスト）を定義

        rb1 = i611Robot()
        rb1.open()

        rb1.move(pickup1)                               # pickup1へPTP動作
        for idx, pos in enumerate(pos_list):            # pos_list[]のデータをを順番にposに取り出し
            rb1.move(pos)
        rb1.move(joint_list[0])                         # joint_list[0]へPTP動作
        rb1.move(joint_list[1])

    Attributes:
        属性の名前 (属性の型): 属性の説明
        属性の名前 (:obj:`属性の型`): 属性の説明.

    """

    #SUFFIX_JSON = '.json'
    #SUFFIX_JSON = '.'                                   # 2020.11.20 従来仕様に合わせて、拡張子無しとする
    SUFFIX_JSON = ''                                   # 2021.02.02 従来仕様に合わせて、拡張子無しとする
    filepath = ""
    filename = ""
    tchFilename = ""
    posName = ""
    teachData = 0

    # ---------------------------------------------------------------------
    def __init__(self):
        """ ZeusTeach Classコンストラクタの説明
        　実行中のスタック情報からZeusTeach()クラスを呼び出した python programの
        　Source file名を取り出し、拡張子を'.JSON'にしたTeaching file名を得る。
        　Teachine fileをOpenして、Dictionary形式のデータ（self.teachData）に読み出す。

        """

        # get Teaching file name
        frame = inspect.stack()[1]                                  # inspect.FrameInfo
        module = inspect.getmodule(frame[0])                        # frame[0] :
        self.filepath = module.__file__                             # source file path
        self.filename = os.path.basename(self.filepath)             # source file name
        st, ext = os.path.splitext(self.filename)                   # "name" + ext
        self.tchFilename = st + self.SUFFIX_JSON                    # "name".json
        # open Teaching data file & read Teaching data file
        self.ztf = ZeusTeachFile(self.tchFilename)                  # Instance generated by specifying the teaching file name
        self.teachData = self.ztf.readDict()                        # Read in the Dictgionary format
        #print(self.teachData)

    # ---------------------------------------------------------------------
    def TPosition(self, dim=0):
        """ TPosition() 関数の説明
        　Position型の teaching data 変数や配列を定義する。
        　引数 dim は定義するPosition型のティーチングデータ配列のサイズを指定する。
        　Default=0は配列ではなく、データ1個の変数を意味する。
         コンストラクタで読み出されているティーチングデータファイルのDictionaryデータから
         名前とサイズが一致するデータを読み出して、Position型のデータ（配列の場合はリスト）を作成し、
         値をコピーして返す。

        Args:
            dim (int):      配列のサイズ（0の場合は変数、Default=0）

        Returns:
            Position型の ティーチングデータ（配列の場合はPosition型のリスト）
            該当データなしの場合、None を返す

        Examples:
            zt = ZeusTeach()                        # ZeusTeachクラス
            pickup1     = zt.TPosition()            # Position型の teaching data を定義
            pos_list    = zt.TPosition(5)           # Position型でサイズ=5の teaching data 配列（リスト）を定義
            pos_list1   = zt.TPosition(1)           # Position型でサイズ=1の teaching data 配列（リスト）を定義

        """

        #print("TPosition(dim): dim=", dim)

        # get Position Name
        frame = inspect.currentframe().f_back                               # The outside of the frame object that called this frame
        self.posName = self._getPosName(frame)                              # Taking out a variable (array) name

        # get the teaching data from Teaching data file
        array = -1 if dim <= 0 else dim-1
        positionData = self.ztf.get_dictPosRec(self.posName, array)         # Taking out a Record of the variable named 'posName'
        #print("positionData=", positionData)

        if positionData is None:
            # ToDo: エラー処理を見直す必要がある
            print("ERROR: Teach data not find !!  ... posName=", self.posName, "dim=", dim)
            liberr(4, 12, "ZeusTeach.TPosition")                            # (4,12):("Key not found.", "", 4),
            return None
        else:
            if array == -1:
                self.pos = copy.deepcopy(self.ztf.getPosition(positionData))
                return self.pos
            else:
                self.poslist = []
                for i in range(dim):
                    positionData = self.ztf.get_dictPosRec(self.posName, i)  # Taking out a Record of the variable named 'posName'
                    self.poslist.append(copy.deepcopy(self.ztf.getPosition(positionData)))
                return self.poslist

    # ---------------------------------------------------------------------
    def TJoint(self, dim=0):
        """ TJoint() 関数の説明
        　Joint型の teaching data 変数や配列を定義する。
        　引数 dim は定義するJoint型のティーチングデータ配列のサイズを指定する。
        　Default=0は配列ではなく、データ1個の変数を意味する。
         コンストラクタで読み出されているティーチングデータファイルのDictionaryデータから
         名前とサイズが一致するデータを読み出して、Joint型のデータ（配列の場合はリスト）を作成し、
         値をコピーして返す。

        Args:
            dim (int):      配列のサイズ（0の場合は変数、Default=0）

        Returns:
            Joint型の ティーチングデータ（配列の場合はJoint型のリスト）
            該当データなしの場合、None を返す

        Examples:
            zt = ZeusTeach()                    # ZeusTeachクラス
            jhome1      = zt.TJoint()           # Joint型の teaching data を定義
            jnt_list    = zt.TJoint(5)          # Joint型でサイズ=5の teaching data 配列（リスト）を定義
            jnt_list1   = zt.TJoint(1)          # Joint型でサイズ=1の teaching data 配列（リスト）を定義

        """

        #print("TJoint(dim): dim=", dim)

        # get Position Name
        frame = inspect.currentframe().f_back                           # The outside of the frame object that called this frame
        self.posName = self._getPosName(frame)                          # Taking out a variable (array) name

        # get the teaching data from Teaching data file
        array = -1 if dim <= 0 else dim - 1
        positionData = self.ztf.get_dictPosRec(self.posName, array)     # Taking out a Record of the variable named 'posName'
        #print("positionData=", positionData)

        if positionData is None:
            # ToDo: エラー処理を見直す必要がある
            print("ERROR: Teach data not find !!  ... posName=", self.posName, "dim=", dim)
            liberr(4, 12, "ZeusTeach.TJoint")                           # (4,12):("Key not found.", "", 4),
            return None
        else:
            if array == -1:
                self.pos = copy.deepcopy(self.ztf.getJoint(positionData))
                return self.pos
            else:
                self.poslist = []
                for i in range(dim):
                    positionData = self.ztf.get_dictPosRec(self.posName, i)  # Taking out a Record of the variable named 'posName'
                    self.poslist.append(copy.deepcopy(self.ztf.getJoint(positionData)))
                return self.poslist

    # ---------------------------------------------------------------------
    def get_tool(self, tool_num):
        """ get_tool() 関数の説明
         コンストラクタで読み出されているティーチングデータファイルのDictionaryデータから
         tool_num（1～8）で指定されるtool_offsetデータを読み出して、次の形式のリストを返す。
                [ index, dx, dy, dz, drz, dry, drx]
                    index :             (int)   :   1..8
                    dx, dy, dz :        (float) :   [mm]
                    drz, dry, drx :     (float) :   [deg]

        Args:
            tool_num (int):      Tool番号（1～8）

        Returns:
            Tool_Offsetデータ
            該当データなしの場合、例外（Robot_error）発生

        Examples:
            zt = ZeusTeach()                    # ZeusTeachクラス
            tool_1 = zt.get_tool(1)             # [1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]

        """

        #print("get_tool(tool): tool=", tool_num)

        # get the tool_offset data from Teaching data file
        toolDic = self.ztf.get_dictToolRec(tool_num)

        #print("toolDic=", toolDic)

        if tool_num == 0:
            self.tool = [0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
        elif toolDic is None:
            return liberr(4, 11, "ZeusTeach.get_tool")                  # (4,11):("Index out of range", "", 4),
        else:
            self.tool = [tool_num, toolDic['OfsX'], toolDic['OfsY'], toolDic['OfsZ'],
                         toolDic['OfsRz'], toolDic['OfsRy'], toolDic['OfsRx']]

        #print("self.tool=", self.tool)
        return self.tool


    # ---------------------------------------------------------------------
    def _getPosName(self, frame):
        """ _getPosName() 関数の説明
        　引数で与えられる TPosition()、TJoint()を呼び出したフレームの情報から、
        　Source上で代入文の左辺に当たる変数名（配列名）を取り出し、文字列として返す。

        Args:
            frame (frame):   TPosition(), TJoint()を呼び出したフレーム

        Returns:
            str:  TPosition, TJointの名前　（Source上で、代入文の左辺の変数名）

        """

        #print("_getPosName")
        #print("frame=", frame, type(frame))

        file = os.path.basename(frame.f_code.co_filename)
        lineno = frame.f_lineno
        target_line = linecache.getline(file, int(lineno))
        #print(file, ":", lineno, ":", target_line)
        linecache.clearcache()

        str_list = target_line.split()
        #print(str_list)
        if str_list[1] == "=":
            posName = str_list[0].split('.')[-1]  # 240118 self. 등의 prefix를 제거하기 위함.
            #print("posName=", posName)
        return posName


#==========================================================================================

from datetime import datetime, date, time
import pprint
from i611_MCS import *


#======================================================================
class ZeusTeachFile():
    """ ZeusTeachFile Class の説明
    　JSON形式の Zeus teaching data file を扱うクラス Classで次の機能を提供する
         1. readDict():          Zeus teaching data file を読み取り Dictionary 形式に格納する
         2. writeDict():         Dictionary形式のdataを teaching data file に保存する
         3. get_dictPosRec(p_name, p_array):
                                 Dictionary形式のdataから、引数 p_name（文字列）と p_array（整数）で
                                 指定される 1 Point分の teaching data record を返す
         4. getPosition(PosRec): 入力の 1 Point分の Dictionary形式の内部データからPosition型のデータを作成
         5. getJoint(PosRec):    入力の 1 Point分の Dictionary形式の内部データからJoint型のデータを作成
         6. readDataFrame():     Zeus teaching data file を読み取り Pandas の DataFrame 形式に格納する
         7. writeDataFrame():    Pandas DataFrame形式のdataを teaching data file に保存する
         8. get_dfTeachData():   pandas DataFrame形式の内部データのうち、 self.dfTeachData を返す
         9. get_dfFrameOffset(): pandas DataFrame形式の内部データのうち、 self.dfFrameOffset を返す
        10. get_dfToolOffset():  pandas DataFrame形式の内部データのうち、 self.dfToolOffset を返す

    Attributes:
        属性の名前 (属性の型): 属性の説明
        属性の名前 (:obj:`属性の型`): 属性の説明.

    """

    LBL_VERSION         = 'version'
    LBL_TEACH_DATA      = 'teach_data'
    LBL_FRAME_OFFSET    = 'frame_offset'
    LBL_TOOL_OFFSET     = 'tool_offset'

    STR_VERSION_1       = '1.0'

    #---------------------------------------------------------------------
    def __init__(self, inTeachfile, outTeachfile=None):
        """ ZeusTeachFile Classコンストラクタの説明

        Args:
            inTeachfile (str): 読み出すZeus teaching data file（省略不可）
            outTeachfile (str、optional): 書き出すZeus teaching data file（省略可）
                                            省略した場合は inTeachfile に書き込みを行う
        """
        scriptDir = os.getcwd()
        self.inTeachfile = os.path.join(scriptDir, inTeachfile)
        if os.path.exists(self.inTeachfile) == False:
            return liberr(4,36)
        self.outTeachfile = self.inTeachfile if outTeachfile is None else os.path.join(scriptDir, outTeachfile)

    #---------------------------------------------------------------------
    def readDict(self):
        """ readDict() 関数の説明
        　コンストラクタで指定した Zeus teaching data file（inTeachfile）を読み込み、
        　Dictionary形式にして内部データの self.dictTD に保存する。

        Returns:
            dictionary: 読み出した self.dictTD
        """
        self.dictTD = json.load(open(self.inTeachfile))
        return self.dictTD

    #---------------------------------------------------------------------
    def writeDict(self):
        """ writeDict() 関数の説明
        　Dictionary形式の内部データ self.dictTD を
        　コンストラクタで指定した Zeus teaching data file（outTeachfile）に出力する。
        """

        with open(self.outTeachfile, 'w') as f:
            f.write(json.dumps(self.dictTD, indent=4))
            #os.fsync(f.fileno())

    #---------------------------------------------------------------------
    def get_dictPosRec(self, p_name, p_array):
        """ get_dictPosRec() 関数の説明
        　Dictionary形式の内部データのうち、引数 p_name（文字列）と p_array（整数）で
          指定される 1 Point分の teaching data record を返す。

        Args:
            p_name (str):   Teaching data の名前
            p_array (int):  Teaching data の’Array'の値　（配列のIndex(0:)、変数は-1）

        Returns:
            見つかった場合：
                dictionary: 1 Point分の teaching data record
            該当なしの場合：
                None

        Examples:
            ztf = ZeusTeachFile("sample.json")                  # teaching file nameを指定して Instance生成
            ztf.readDict()                                      # Dictgionary形式に読み出し

            rec = ztf.get_dictPosRec('pickup2', -1)             # 'pickup2'という名前の変数のRecordを取り出す
            rec['VarInt'] = 10                                  # 'VarInt' のデータを 10 に書き換え

            ztf.writeDict()                                     # teaching file 更新

        Note:
            dictionary['teach_data']の 1 Point分の teaching data record
                'key' (型):             説明                                      例
            -------------------------------------------------------------
                "idx" (int):            データのIndex （1から）                     1
                "PositionName" (str):   変数、配列の名前                           "pickup1"
                "Array" (int):          配列のIndex　（０から、変数は-1）            -1
                "DataType" (int):       dataのType　（次の整数値）                  1
                    （Initial、Undef＝０、Position＝１、Joint＝２、Pulse＝３、Int＝４、Float＝５）
                "Replace" (int):        Replace実施済み` （0:未実施 / 1:実施済）     0
                "DateTime" (str):       Replace日時                              "2020/10/21 10:25:52"
                "PosX" (float):         XY座標系の X座標値 [mm]                    2.0
                "PosY" (float):         XY座標系の Y座標値 [mm]                    0.0
                "PosZ" (float):         XY座標系の Z座標値 [mm]                    0.0
                "PosRz" (float):        XY座標系の Rz座標値 [deg]                  0.0
                "PosRy" (float):        XY座標系の Ry座標値 [deg]                  0.0
                "PosRx" (float):        XY座標系の Rx座標値 [deg]                  0.0
                "PosPosture" (int):     XY座標系の Postureの値                     0
                "PosMulti" (int):       XY座標系の MultiTurnの値                   0
                "Jnt1" (float):         Joint座標系の J1座標値 [deg]               1.0
                "Jnt2" (float):         Joint座標系の J2座標値 [deg]               2.0
                "Jnt3" (float):         Joint座標系の J3座標値 [deg]               3.0
                "Jnt4" (float):         Joint座標系の J4座標値 [deg]               4.0
                "Jnt5" (float):         Joint座標系の J5座標値 [deg]               5.0
                "Jnt6" (float):         Joint座標系の J6座標値 [deg]               6.0
                "Pls1" (int):           J1軸のパルス数 [pulse]                     3
                "Pls2" (int):           J2軸のパルス数 [pulse]                     0
                "Pls3" (int):           J3軸のパルス数 [pulse]                     0
                "Pls4" (int):           J4軸のパルス数 [pulse]                     0
                "Pls5" (int):           J5軸のパルス数 [pulse]                     0
                "Pls6" (int):           J6軸のパルス数 [pulse]                     0
                "VarInt" (int):         Int型のデータ値                            0
                "VarFloat" (float):     Float型のデータ値                          0.0
                "FrameNo" (int):        Frame使用時の Frame番号                    0
                "ToolNo" (int):         Tool使用時の Tool番号                      0
                "RobotNo" (int):        Robot番号                                1
                "Description" (str):    Positionデータの説明                       ""
        """

        for d in self.dictTD['teach_data']:
            name = d.get('PositionName')
            array = d.get('Array')
            if (name == p_name) and (array == p_array):
                return d
        return None

    #---------------------------------------------------------------------
    def get_dictToolRec(self, p_index):
        """ get_dictPosRec() 関数の説明
        　Dictionary形式の内部データのうち、引数 p_index（整数）で
          指定される 1個分の tool offset data record を返す。

        Args:
            p_index (int):  tool_offset data の 'idx'の値　（配列のIndex(1:8)）

        Returns:
            見つかった場合：
                dictionary: 1個分の tool_offset data record
            該当なしの場合：
                None

        Examples:
            ztf = ZeusTeachFile("sample.json")                  # teaching file nameを指定して Instance生成
            ztf.readDict()                                      # Dictgionary形式に読み出し

            rec = ztf.get_dictToolRec(1)                        # idx=1 のtool_offset Recordを取り出す
            toolOfsX = rec['OfsX']                              # 'OfsX' のデータ（float）を取り出し

        Note:
            dictionary['tool_offset']の 1個分の tool_offset data record
                'key' (型):             説明                                      例
            -------------------------------------------------------------
                "idx" (int):            データのIndex （1から8）                   1
                "RobotNo" (int):        Robot番号                                  1
                "Description" (str):    Positionデータの説明                       ""
                "OfsX" (float):         ToolOffsetの X-offset値 [mm]               0.0
                "OfsY" (float):         ToolOffsetの Y-offset値 [mm]               0.0
                "OfsZ" (float):         ToolOffsetの Z-offset値 [mm]               0.0
                "OfsRz" (float):        ToolOffsetの Rz-offset値 [deg]             0.0
                "OfsRy" (float):        ToolOffsetの Ry-offset値 [deg]             0.0
                "OfsRx" (float):        ToolOffsetの Rx-offset値 [deg]             0.0
        """

        #print("get_dictToolRec(p_index): p_index=", p_index)


        for d in self.dictTD['tool_offset']:
            idx = int(d.get('idx'))

            #print('idx, d=', idx, vars(d))
            #print('idx=', idx)

            if (idx == p_index):
                return d
        return None

    # ---------------------------------------------------------------------
    def getPosition(self, PosRec):
        """ getPosition() 関数の説明
        　入力の 1 Point分の Dictionary形式の内部データ（teaching data record）の
        　次の情報から、Position型のデータを作成して返す
                "PosX" (float):         XY座標系の X座標値 [mm]                    2.0
                "PosY" (float):         XY座標系の Y座標値 [mm]                    0.0
                "PosZ" (float):         XY座標系の Z座標値 [mm]                    0.0
                "PosRz" (float):        XY座標系の Rz座標値 [deg]                  0.0
                "PosRy" (float):        XY座標系の Ry座標値 [deg]                  0.0
                "PosRx" (float):        XY座標系の Rx座標値 [deg]                  0.0
                "PosPosture" (int):     XY座標系の Postureの値                     0
                "PosMulti" (int):       XY座標系の MultiTurnの値                   0

        Args:
            PosRec (dictionary):   1 Point分の Dictionary形式の内部データ

        Returns:
            Position:   Position型の position data

        Examples:
            ztf = ZeusTeachFile("sample.json")                  # teaching file nameを指定して Instance生成
            ztf.readDict()                                      # Dictgionary形式に読み出し
            rec = ztf.get_dictPosRec('pickup2', -1)             # 'pickup2'という名前の変数のRecordを取り出す
            posData = ztf.getPosition(rec)                      # Position型のデータにして取り出す

        """

        tpos = Position(x=PosRec['PosX'], y=PosRec['PosY'], z=PosRec['PosZ'],
                        rz=PosRec['PosRz'], ry=PosRec['PosRy'], rx=PosRec['PosRx'],
                        posture=PosRec['PosPosture'], multiturn=PosRec['PosMulti'])
        return tpos

    # ---------------------------------------------------------------------
    def getJoint(self, PosRec):
        """ getJoint() 関数の説明
        　入力の 1 Point分の Dictionary形式の内部データ（teaching data record）の
        　次の情報から、Joint型のデータを作成して返す
                "Jnt1" (float):         Joint座標系の J1座標値 [deg]               1.0
                "Jnt2" (float):         Joint座標系の J2座標値 [deg]               2.0
                "Jnt3" (float):         Joint座標系の J3座標値 [deg]               3.0
                "Jnt4" (float):         Joint座標系の J4座標値 [deg]               4.0
                "Jnt5" (float):         Joint座標系の J5座標値 [deg]               5.0
                "Jnt6" (float):         Joint座標系の J6座標値 [deg]               6.0

        Args:
            PosRec (dictionary):   1 Point分の Dictionary形式の内部データ

        Returns:
            Joint:  Position型の position data

        Examples:
            ztf = ZeusTeachFile("sample.json")                  # teaching file nameを指定して Instance生成
            ztf.readDict()                                      # Dictgionary形式に読み出し
            rec = ztf.get_dictPosRec('pickup2', -1)             # 'pickup2'という名前の変数のRecordを取り出す
            jntData = ztf.getJoint(rec)                         # Joint型のデータにして取り出す

        """

        tpos = Joint(j1=PosRec['Jnt1'], j2=PosRec['Jnt2'], j3=PosRec['Jnt3'],
                        j4=PosRec['Jnt4'], j5=PosRec['Jnt5'], j6=PosRec['Jnt6'])
        return tpos



if __name__ == '__main__':
    zt         = ZeusTeach()
    Home_joint = zt.TJoint(10) 
    pickup1    = zt.TPosition()
    pickup2    = zt.TJoint()
    pickup3    = zt.TPosition()
    pickup4    = zt.TJoint()
    pickup5    = zt.TPosition(2)
    pickdown1  = zt.TPosition(10)
    pickdown2  = zt.TPosition(2)
    pickdown3  = zt.TPosition(2)
    pickdown4  = zt.TPosition(2)
    pickdown5  = zt.TPosition(2)

    a          = zt.TPosition(30)
    b          = zt.TJoint(30)

 
    print('pickup1=', vars(pickup1))
    print('pickup2=', vars(pickup2))
    print('pickup3=', vars(pickup3))
    print('pickup4=', vars(pickup4))

    print('pickup5=', pickup5)
    for idx, pos in enumerate(pickup5):
        print('pickup5[', idx, ']=', vars(pos))


    for i in range(5):
        print(a[i].pos2list())
        print(b[i].jnt2list())



    """
    rb1 = i611Robot()
    rb1.open()
    
    print('pickup1=', vars(pickup1))
    rb1.move(pickup1)

    print('pickup2=', vars(pickup2))
    rb1.move(pickup2)

    print('pickup3=', vars(pickup3))
    rb1.move(pickup3)

    print('pickup4=', vars(pickup4))
    rb1.move(pickup4)

    for idx, pos in enumerate(pickup5):
        print('pos=', vars(pos))
        rb1.move(pos)

    print('pickup5[0]=', vars(pickup5[0]))
    rb1.move(pickup5[0])

    print('pickup5[1]=', vars(pickup5[1]))
    rb1.move(pickup5[1])

    """
