#!/usr/bin/env python
# -*- coding: utf-8 -*-


import threading
import subprocess
import os
import sys
import json
import time
from collections import Counter
import socket,re
import datetime
import traceback

import i611_common
from rblib import Robot
from i611shm import shm_system_write, shm_read
from i611_common import Robot_error
from i611_MCS import i611Robot

FNAME_INIT      = '/home/i611usr/init.py'
FNAME_IOSTAT    = '/tmp/iostat'
FNAME_AUTOREADY = '/tmp/auto_ready'
FNAME_TEACH_PID = '/tmp/i611_teach.pid'
FNAME_JOG_MODE  = '/tmp/jog_mode'
FNAME_JOG_PROC  = '/usr/lib/python2.7/site-packages/i611jog.py'
FNAME_ZEUSTEACH = '/tmp/zeus_teach'                                     # CT210121

SYSINFO_ADDR1   = 0x0800
SYSINFO_ADDR2   = 0x0844

#[Public] ############################
def version_i611sys():
    return [0, 3, 5, 0]

"""======================================================================
[内部クラス]
  _commTask() : robタスク、IOタスクと通信
  host = 127.0.0.1 , socket = 4000 , listen = 10 , buff = 256 , wait = ?
  
  setRecvData(self, data):	rbsysとのI/F
======================================================================"""
class _commTask(i611_common._SockServer):
    def __init__(self, sysstatus):
        super(_commTask, self).__init__()
        self.__sysstatus = sysstatus

    # [Internal] Remote api handler
    def setRecvData(self, data):
        jsonobj = json.loads(data)

        if jsonobj['cmd'] == 1: # set_robtask
            params = jsonobj['params']
            res = self.__sysstatus.handle_set_robtask(params[0])
            result = {'cmd':1, 'results':res}

        elif jsonobj['cmd'] == 2: # get_robtask
            res = self.__sysstatus.handle_get_robtask()
            result = {'cmd':2, 'results':res}

        elif jsonobj['cmd'] == 3: # clear_robtask
            res = self.__sysstatus.handle_clear_robtask()
            result = {'cmd':3, 'results':res}

        elif jsonobj['cmd'] == 4: # req_mcmd
            params = jsonobj['params']
            res = self.__sysstatus.handle_req_mcmd()
            result = {'cmd':4, 'results':res}

        elif  jsonobj['cmd'] == 8: # cmd_run    (=run input)
            params = jsonobj['params']
            res = self.__sysstatus.handle_cmd_run(params[0])
            result = {'cmd':8, 'results':res}

        elif  jsonobj['cmd'] == 13: # cmd_stop  (=stop input)
            res = self.__sysstatus.handle_cmd_stop()
            result = {'cmd':13, 'results':res}

        elif  jsonobj['cmd'] == 14: # cmd_reset (=reset input)
            res = self.__sysstatus.handle_cmd_reset()
            result = {'cmd':14, 'results':res}

        elif  jsonobj['cmd'] == 21: #  cmd_pause (=pause input)
            res = self.__sysstatus.handle_cmd_pause()
            result = {'cmd':21, 'results':res}

        elif  jsonobj['cmd'] == 17: # assign_din
            params = jsonobj['params']
            res = self.__sysstatus.handle_assign_din(params[0:4])
            result = {'cmd':17, 'results':res}

        elif  jsonobj['cmd'] == 18: # assign_dout
            params = jsonobj['params']
            res = self.__sysstatus.handle_assign_dout(params[0:8])
            result = {'cmd':18, 'results':res}

        elif  jsonobj['cmd'] == 20: # version
            res = version_i611sys()
            result = {'cmd':20, 'results':res}

        elif  jsonobj['cmd'] == 22: # register_program
            params = jsonobj['params']
            res = self.__sysstatus.handle_cmd_register_program(params[0])
            result = {'cmd':22, 'results':res}

        elif  jsonobj['cmd'] == 23: # debug command
            params = jsonobj['params']
            res = self.__sysstatus.handle_cmd_debug(params[0])
            result = {'cmd':23, 'results':res}

        elif  jsonobj['cmd'] == 24: # internal call
            params = jsonobj['params']
            res = self.__sysstatus.handle_cmd_internal_call(params[0:2])
            result = {'cmd':24, 'results':res}

        else:
            result = {'cmd':0, 'results':[]}


        time.sleep(0)
        self.setSendData(json.dumps(result))

def print2(msg):
    sys.stdout.write(msg+"\n")
    sys.stdout.flush()

"""======================================================================
[内部クラス]
  _watchIO(host, port, stasus) :
  IO監視
======================================================================"""
class _watchIO(threading.Thread):
    def __init__(self, host, port, status):
        super(_watchIO, self).__init__()
        self.setDaemon(True)
        self.__flg_stop = threading.Event()

        self.__sts = status
        self.__last_indicator_data = 0x00000000
        self.__rblib = Robot(host, port)
        self.__rblib.open()
        self.__sts._rblib = self.__rblib
        #self.__rblib.sysctrl(1,0)
        self.__sts.is_simulator = os.path.exists('/etc/mcs/SIM_MODE')
        # check serve (only real mode)
        if not self.__sts.is_simulator:
            ret = self.__rblib.ioctrl(128, 0, 0xffffffff, 0, 0xffffffff)
            if ret[0] == False or ret[1] & 0x01 != 0:
                print "Force servo off"
                ret = self.__rblib.acq_permission()
                if ret[0] == False:
                    raise Robot_error("Unable to turn servo off!!(1)")
                ret = self.__rblib.svctrl(2)  # svoff myself
                if ret[0] == False:
                    raise Robot_error("Unable to turn servo off!!(1)")
                ret = self.__rblib.rel_permission()
                if ret[0] == False:
                    raise Robot_error("Unable to turn servo off!!(1)")
#                print "boot check=",ret
#                raise Robot_error("ERROR! Servo must be off when system manager is booting!!")
        self.toggle_write_extend_port = 0
        self.last_extend_port = -1
        self.last_auto_mode = False
        self.jog_proc = None
        self.output_slave_info_done = False


    def __del__(self):
        self.__rblib.close()

    def thread_stop(self):
        self.__flg_stop.set()
        self.join(1)

    def run(self):
        try:                # スレッドだけ終了してしまう問題の対策
            self.run_sub()  # Actual thread func
        except Exception:
            sys.stderr.write("_watchIO caused exception\n")
            traceback.print_exc()
            os._exit(1)

    def run_sub(self):
        # warning once flag
        jog_warn_once = False
        # check backup result once
        backup_check_once = False

        self.__rblib.sysctrl(1,0)

        # initialize memory I/O
        self.write_memory_io_word(130, 0, 0x00000000)

        # $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$
        # System Manager Main Loop
        while not self.__flg_stop.is_set():

            ## outout slave info once
            if not self.output_slave_info_done:
                if os.path.exists("/opt/i611/fwver"):
                    self.__sts.output_slave_info()
                    self.output_slave_info_done = True

            ## backup result check
            if not backup_check_once:
                if os.path.exists("/tmp/backup_result"):
                    time.sleep(0.5)
                    with open("/tmp/backup_result") as f:
                        val = f.readline()
                        if val != "":
                            result = int(val)
                            if result == 0:
                                self.__sts.status_log(self.__sts.TAG_EVENT,"Backup home OK")
                            elif result == 1: # home is too large
                                self.__sts.set_system_status(self.__sts.SS_NERR, 53)  # E53
                            elif result == 2: # free is too low
                                self.__sts.set_system_status(self.__sts.SS_CERR, 6) # C06

                    backup_check_once = True

            ## Read In Ports (pulse)
            # PortG1 [ 0:svon,      1:emo,      2:hw_error,     3:abs_lost  ] In/Out
            st_port_value = self.read_memory_io_word(128)
            for i in range(self.__sts.G1_PORT_NUM):
                cmd = 1 if (st_port_value& (1<<i)) != 0 else 0
                self.__sts.st_preset_in_G1[i] = cmd
            # HW error code
            self.__sts.hw_err_code = (st_port_value >> 8) & 0x0FF
            if self.__sts.st_preset_in_G1[self.__sts.G1_HW_ERROR] == 1:
                if self.__sts.hw_err_code == 0:
                    print "Native Invalid Bit Error=%08X" % st_port_value
                #else:
                    #print "Native Normal Bit Error=%08X" % st_port_value

            # PortG3 [ 0:run,       1:stop,     2:err_reset,    3:pause     ] In
            st_port_value = self.read_memory_io_word(134)
            st_phyio_value = self.read_memory_io_word(0)
            for i in range(self.__sts.G3_PORT_NUM):
                cmd = 1 if (st_port_value& (1<<i)) != 0 else 0
                self.__sts.st_preset_in_G3[i] = self.__sts.detect_preset_in_G3[i].pls(cmd)
                if self.__sts.id_assign_in_G3[i] != -1:
                    if self.__sts.id_assign_in_G3[i] < 32:
                        cmd = 1 if (st_phyio_value & (1<<self.__sts.id_assign_in_G3[i])) != 0 else 0
                    else:
                        cmd = self.read_memory_io_bit(self.__sts.id_assign_in_G3[i])
                    self.__sts.st_assign_in_G3[i] = self.__sts.detect_assign_in_G3[i].pls(cmd)

            ## Read App State
            st_port_value = self.read_memory_io_word(130)
            self.__sts.app_exit_code = (st_port_value >> 16) &    0x00FF
            self.__sts.app_run_status = (st_port_value >> 24) &   0x000F
            self.__sts.disable_io_input = (st_port_value >> 28) & 0x0001
            self.__sts.app_pause_status = (st_port_value >> 29) & 0x0007

            ## Read Extend In Port ( Mode(Large), Door(Large), ID )
            self.__sts.st_extend_port = self.read_extend_ports()


            ## Read other info
            self.__sts.exists_auto_ready = True if os.path.exists('/tmp/auto_ready') else False
            #flag_detect_jog = True if os.path.exists('/tmp/detect_jog') else False

            ## force I/O event
            if self.__sts.st_preset_in_G1[self.__sts.G1_HW_ERROR] == 1:
                self.__sts.set_system_status(self.__sts.SS_CERR, self.__sts.hw_err_code)
            elif (self.__sts.system_status != self.__sts.SS_JOG and
                self.__sts.st_preset_in_G1[self.__sts.G1_ABS_LOST] == 1):
                if self.__sts.system_status < self.__sts.SS_NERR:
                    self.__sts.set_system_status(self.__sts.SS_INC)

            ## Go to jog mode
            if (self.__sts.system_status != self.__sts.SS_JOG and self.__sts.system_status < self.__sts.SS_NERR
                and os.path.exists(FNAME_JOG_MODE) ):
                print "launch jog process"
                self.launch_jog_process()
                self.__sts.set_system_status(self.__sts.SS_JOG)

            ## Command Handler ###########################
            # copy command to local
            robsys_cmd = self.__sts.command_from_robsys
            self.__sts.command_from_robsys = self.__sts.CMD_RBSYS_NONE

            # [RUN]
            if (robsys_cmd == self.__sts.CMD_RBSYS_RUN or
                self.__sts.st_preset_in_G3[self.__sts.G3_RUN] == 1 or
                self.__sts.st_assign_in_G3[self.__sts.G3_RUN] == 1):
                if self.__sts.disable_io_input == 0: # if I/O input is enabled
                    if self.__sts.system_status == self.__sts.SS_READY:
                        self.__sts.now_pause = 0
                        self.__sts.request_to_app = 0
                        self.set_app_status_to_run()
                        ret = self.__sts.exec_cmd_run() # ***
                        if ret != 0:
                            self.__sts.set_system_status(self.__sts.SS_NERR, ret)
                    elif self.__sts.system_status == self.__sts.SS_INC:
                        self.__sts.set_system_status(self.__sts.SS_NERR, 7)
                    else:
                        print "Ignore run cmd because of invalid mode"

            # [Register from app]
            if robsys_cmd == self.__sts.CMD_RBSYS_REGISTER:
                self.__sts.now_pause = 0
                self.__sts.request_to_app = 0
                ##XX self.set_app_status_to_run()
                self.__sts.set_system_status(self.__sts.SS_RUN)


            ## Judge Status and Event ###########################
            if self.__sts.system_status == self.__sts.SS_INIT:
                pass    # can do nothing

            elif self.__sts.system_status == self.__sts.SS_READY:

                if os.path.exists(FNAME_TEACH_PID):
                    jog_warn_once = False
                    ft = open(FNAME_TEACH_PID,"r")
                    pid = ft.readline()
                    ft.close()
                    if pid != "":
                        proc_res = self.__sts.proc_poll(int(pid))
                        if proc_res == 1:
                            self.__sts.teach_pid = int(pid)
                            self.__sts.set_system_status(self.__sts.SS_TEACH)
                            self.__sts.is_ZeusTeach = False		                # CT210121
                        else: # teach error!
                            os.remove(FNAME_TEACH_PID)
                            self.__sts.set_system_status(self.__sts.SS_NERR,40)  # NERR(40)

                elif os.path.exists(FNAME_ZEUSTEACH):                           # CT210121
                    self.__sts.set_system_status(self.__sts.SS_TEACH)           # CT210121
                    self.__sts.is_ZeusTeach = True      		                # CT210121

            elif self.__sts.system_status == self.__sts.SS_INC:
                pass    # can do nothing

            elif self.__sts.system_status == self.__sts.SS_TEACH:
                if self.__sts.is_ZeusTeach == True:      		                # CT210121
                    if not os.path.exists(FNAME_ZEUSTEACH):                     # CT210121
                        print "zeus teach finished"                             # CT210121
                        self.__sts.set_system_status(self.__sts.SS_READY)       # CT210121
                        self.__sts.is_ZeusTeach = False		                    # CT210121
                else:                                                           # CT210121
                    proc_res = self.__sts.proc_poll(self.__sts.teach_pid )
                    if proc_res <= 0:
                        print "teach finished (%d)" % self.__sts.teach_pid
                        self.__sts.set_system_status(self.__sts.SS_READY)   # back to ready

                    # Check teach condition
                    if jog_warn_once == False and not os.path.exists('/tmp/detect_jog'):
                        mid = (self.__sts.st_extend_port >> 0) & 0x03
                        door = (self.__sts.st_extend_port >> 2) & 0x03
                        mode = (self.__sts.st_extend_port >> 4) & 0x03
                        jpcn = (self.__sts.st_extend_port >> 6) & 0x01
                        msg = "Unplug JOG during teach,ID=%d,Door=%d,Mode=3>%d,jpcn=0>%d" % (
                            mid,door,mode,jpcn)
                        self.__sts.warning_log(msg)
                        jog_warn_once = True

            elif self.__sts.system_status == self.__sts.SS_JOG:
                if self.jog_proc is not None:
                    jog_st = self.jog_proc.poll()                           # child process state
                else:
                    jog_st = -1
#                if jog_st != None or cur_status != self.__sts.SS_JOG:
                if jog_st is not None:
                    self.__sts.set_system_status(self.__sts.SS_CERR,4)   # Critical error during JOG mode
                    self.jog_proc = None
                    self.__rblib.sysctrl(0x8000,0xdead)                  # lock native

            elif self.__sts.system_status == self.__sts.SS_RUN or self.__sts.system_status == self.__sts.SS_PAUSE:
                if self.__sts.app_run_status == self.__sts.APP_PAUSE:
                    self.__sts.now_pause = 1
                    if self.__sts.system_status == self.__sts.SS_RUN:
                        self.__sts.set_system_status(self.__sts.SS_PAUSE)
                else:
                    self.__sts.now_pause = 0
                    if self.__sts.system_status == self.__sts.SS_PAUSE:
                        self.__sts.set_system_status(self.__sts.SS_RUN)

                prog_state = self.__sts.proc_poll(self.__sts._prog_pid)
                if prog_state == 1:  # stil running
                    pass
                elif prog_state <= 0:  # judge exit status from shared memory
                    print "detect process exit"
                    self.__sts.prog_wait()
                    self.__sts.clear_rbfname()
                    self.__sts.now_pause = 0
                    self.__sts.request_to_app = 0
                    time.sleep(0.5)
                    # call abort just in case
                    self.__sts.warning_log("i611sys.abortm()")
                    self.__rblib.abortm()

                    ## ReRead App State
                    st_port_value = self.read_memory_io_word(130)
                    self.__sts.app_exit_code = (st_port_value >> 16) &    0x00FF
                    self.__sts.app_run_status = (st_port_value >> 24) &   0x000F
                    self.__sts.disable_io_input = (st_port_value >> 28) & 0x0001
                    print "read app status(%d,%d)" % (self.__sts.app_run_status,
                        self.__sts.app_exit_code)
                    if self.__sts.app_run_status == 0: # abnormal exit
                        self.__sts.set_system_status(self.__sts.SS_NERR,8)  # NERR(8)

                    elif self.__sts.app_run_status == 1: # normal exit
                        self.__sts.set_system_status(self.__sts.SS_READY)

                    elif self.__sts.app_run_status == 4: # exit from pause
                        self.__sts.set_system_status(self.__sts.SS_READY)

                    elif self.__sts.app_run_status == 2: # error exit
                        self.__sts.set_system_status(self.__sts.SS_NERR,self.__sts.app_exit_code)

                    elif self.__sts.app_run_status == 3: # exception exit
                        self.__sts.set_system_status(self.__sts.SS_NERR,self.__sts.app_exit_code)

                    elif self.__sts.app_run_status == self.__sts.APP_EXIT_CRITICAL_ERROR:
                        self.__sts.set_system_status(self.__sts.SS_CERR,self.__sts.app_exit_code)

                    elif self.__sts.app_run_status == self.__sts.APP_EXIT_USER_NORMAL_ERROR:
                        self.__sts.set_system_status(self.__sts.SS_UNERR,self.__sts.app_exit_code)

                    elif self.__sts.app_run_status == self.__sts.APP_EXIT_USER_CRITICAL_ERROR:
                        self.__sts.set_system_status(self.__sts.SS_UCERR,self.__sts.app_exit_code)

                    else: # unknown error
                        self.__sts.set_system_status(self.__sts.SS_NERR,99)

                # command stop
                if (robsys_cmd == self.__sts.CMD_RBSYS_STOP or
                    self.__sts.st_preset_in_G3[self.__sts.G3_STOP] == 1 or
                    self.__sts.st_assign_in_G3[self.__sts.G3_STOP] == 1):
                    if self.__sts.disable_io_input == 0: # if I/O input is enabled
                        self.__sts.exec_cmd_stop() # ***

                # command pause
                elif (robsys_cmd == self.__sts.CMD_RBSYS_PAUSE or
                    self.__sts.st_preset_in_G3[self.__sts.G3_PAUSE] == 1 or
                    self.__sts.st_assign_in_G3[self.__sts.G3_PAUSE] == 1):
                    if self.__sts.now_pause == 0:
                        if self.__sts.disable_io_input == 0: # if I/O input is enabled
                            self.__sts.exec_cmd_pause() # ***

                # command continue(=run)
                elif (robsys_cmd == self.__sts.CMD_RBSYS_RUN or
                    self.__sts.st_preset_in_G3[self.__sts.G3_RUN] == 1 or
                    self.__sts.st_assign_in_G3[self.__sts.G3_RUN] == 1):
                    if self.__sts.now_pause == 1:
                        if self.__sts.disable_io_input == 0: # if I/O input is enabled
                            self.__sts.exec_cmd_continue() # ***

            elif (self.__sts.system_status == self.__sts.SS_NERR or
                self.__sts.system_status == self.__sts.SS_UNERR):
                # command reset
                if (robsys_cmd == self.__sts.CMD_RBSYS_RESET or
                    self.__sts.st_preset_in_G3[self.__sts.G3_ERR_RESET] == 1 or
                    self.__sts.st_assign_in_G3[self.__sts.G3_ERR_RESET] == 1):
                    print "detect_reset!!!"
                    self.__sts.set_system_status(self.__sts.SS_READY)   # Reset Err!!
                    self.set_app_status_to_run()

            elif (self.__sts.system_status == self.__sts.SS_CERR or
                self.__sts.system_status == self.__sts.SS_UCERR):

                if self.__sts.cur_error_no == 99 and self.__sts.hw_err_code != 0:
                     # update c99 error code
                     self.__sts.cur_error_no = self.__sts.hw_err_code
                     self.__sts.set_system_status(self.__sts.SS_CERR,self.__sts.hw_err_code,True)

            else:
                print "Error! unknown status!"
            ####################################################

            ## reflect to Output
            # running
            if self.__sts.system_status == self.__sts.SS_RUN or self.__sts.system_status == self.__sts.SS_PAUSE:
                self.__sts.st_out_G2[self.__sts.G2_RUNNING] = 1
            else:
                self.__sts.st_out_G2[self.__sts.G2_RUNNING] = 0

            # sw_error
            if (self.__sts.system_status == self.__sts.SS_NERR or
                self.__sts.system_status == self.__sts.SS_UNERR):
                self.__sts.st_out_G2[self.__sts.G2_SW_ERROR] = 1
            else:
                self.__sts.st_out_G2[self.__sts.G2_SW_ERROR] = 0

            # in_pause
            if self.__sts.now_pause:
                self.__sts.st_out_G2[self.__sts.G2_IN_PAUSE] = 1
            else:
                self.__sts.st_out_G2[self.__sts.G2_IN_PAUSE] = 0

            # error
            if (self.__sts.system_status >= self.__sts.SS_NERR and
                    self.__sts.system_status <= self.__sts.SS_UCERR):
                self.__sts.st_out_G2[self.__sts.G2_ERROR] = 1
            else:
                self.__sts.st_out_G2[self.__sts.G2_ERROR] = 0

            ## clear command flags
            self.__sts.command_from_rbsys = self.__sts.CMD_RBSYS_NONE

            ## Write Out Ports (pulse)
            # PortG1 [ 0:svon,      1:emo,      2:hw_error,     3:abs_lost  ] In/Out
            st_dio_port_value = 0
            st_dio_mask_value = 0xFFFFFFFF
            for i in range(self.__sts.G1_PORT_NUM):
                port_no = self.__sts.id_assign_out_G1[i]
                port_st = self.__sts.st_preset_in_G1[i]
                if port_no > 31: # memory I/O
                    self.write_memory_io_bit(port_no, port_st)
                elif port_no >= 0: # DIO
                    st_dio_mask_value &= ~(1<<port_no)
                    if port_st == 1:
                        st_dio_port_value |= (1 << port_no)
                    else:
                        st_dio_port_value &= ~(1 << port_no)

            # PortG2 [ 0:running,1:sw_error, 2:in_pause, 3:error,( 4-7:status, 8-15:err code) ] Out
            st_g2_port_state = 0
            for i in range(self.__sts.G2_PORT_NUM):
                port_no = self.__sts.id_assign_out_G2[i]
                port_st = self.__sts.st_out_G2[i]
                if port_no > 31: # memory I/O
                    self.write_memory_io_bit(port_no, port_st)
                elif port_no >= 0: # DIO
                    st_dio_mask_value &= ~(1 << port_no)
                    if port_st == 1:
                        st_dio_port_value |= (1 << port_no)
                    else:
                        st_dio_port_value &= ~(1 << port_no)
                # preset port
                if port_st == 1:
                    st_g2_port_state |= (1 << i)
            # status & err_code
            st_g2_port_state |= (self.__sts.system_status&0x0F)<<4
            st_g2_port_state |= (self.__sts.cur_error_no&0x00FF)<<8
            self.write_memory_io_word(0, st_dio_port_value, st_dio_mask_value)
            self.write_memory_io_word(130, st_g2_port_state, 0xFFFF0000)

            ## inform Jog of Extend Out Port
            if self.last_extend_port != self.__sts.st_extend_port:
                self.write_extend_ports_for_jog(self.__sts.st_extend_port)
                self.last_extend_port = self.__sts.st_extend_port

            ## Write Extend Out Port ( Mode IO(Large & normal) )
            if self.last_auto_mode != self.__sts.exists_auto_ready:
                self.write_mode_io(self.__sts.exists_auto_ready)
                self.last_auto_mode = self.__sts.exists_auto_ready

            # update status log constantly even if no change
            self.__sts.update_status_log()

            # update front indicator
            if self.__last_indicator_data != self.__sts.indicator_data:
                if not self.__sts.system_status == self.__sts.SS_JOG:
                    self.write_memory_io_word(4,self.__sts.indicator_data,0x000000DF)
                self.__last_indicator_data = self.__sts.indicator_data

            ## Loop wait (50ms)
            time.sleep(0.02)

        # System Manager Main Loop End
        # $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$

    # [Internal] launch Jog mode
    def launch_jog_process(self):
        FILE_OUT_LOG = '/opt/i611/log/jog_out.log'
        FILE_ERR_LOG = '/opt/i611/log/jog_err.log'
        if os.path.exists(FILE_ERR_LOG):
            if os.path.getsize(FILE_ERR_LOG) > 10*1024*1024: # 10MB
                os.remove(FILE_ERR_LOG)
        # create log file
        fout = open(FILE_OUT_LOG,'w')
        ferr = open(FILE_ERR_LOG,'a')
        now = datetime.datetime.today()
        timestr = "\nBegin:%04d%02d%02d %02d:%02d:%02d.%03d\n" % (now.year,
            now.month, now.day, now.hour, now.minute, now.second, now.microsecond/1000)
        fout.write(timestr)
        ferr.write(timestr)
        fout.close()
        ferr.close()
        fout = os.open(FILE_OUT_LOG,os.O_WRONLY|os.O_APPEND)
        ferr = os.open(FILE_ERR_LOG,os.O_WRONLY|os.O_APPEND)
        self.jog_proc = subprocess.Popen(('python', FNAME_JOG_PROC),stdout=fout,stderr=ferr)

    # [Internal] read hardware id [ID2:ID1]
    def detect_hardware(self):
        val = self.read_extend_ports()
        val &= 0x03
        self.__sts.hardware_id = val
        return val

    # [Internal] wait until native is ready
    def wait_until_ready(self):
        while True:
            ret = self.__rblib.syssts(8)
            if ret[0] == False:
                return False
            if ret[1] != 0:
                return True
            time.sleep(0.5)

    # [Internal] read extend port for large robo
    def read_extend_ports(self):
        BIT11_REMOTE2 = 0x0800
        BIT10_REMOTE1 = 0x0400
        BIT9_ID3      = 0x0100
        BIT8_ID2      = 0x0200
        #BIT7_NOUSE   = 0x0080
        BIT6_JPCN     = 0x0040
        BIT5_MODE2    = 0x0020
        BIT4_MODE1    = 0x0010
        BIT3_DOOR2    = 0x0008
        BIT2_DOOR1    = 0x0004
        BIT1_ID1      = 0x0002
        BIT0_ID0      = 0x0001
        flag = 0
        bits = self.read_memory_io_word(3) #IN_RH/IN_RL
        if bits & (0x01 << 17):  # ID1 (col=3, bit=17)
            flag |= BIT1_ID1
        if bits & (0x01 << 16):  # ID0 (col=3, bit=16)
            flag |= BIT0_ID0
        if bits & (0x01 << 29):   # REMOTE1 (col=1, bit=5)
            flag |= BIT10_REMOTE1
        if bits & (0x01 << 30):   # REMOTE2 (col=1, bit=6)
            flag |= BIT11_REMOTE2

        bits = self.read_memory_io_word(5) #IN_X1/IN_X2
        if bits & (0x01 << 10):  # JPCN  (col=5, bit=10)
            flag |= BIT6_JPCN
        if bits & (0x01 << 13):  # DOOR2 (col=5, bit=13)
            flag |= BIT3_DOOR2
        if bits & (0x01 << 12):  # DOOR1 (col=5, bit=12)
            flag |= BIT2_DOOR1
        if bits & (0x01 << 15):  # MODE2 (col=5, bit=15)
            flag |= BIT5_MODE2
        if bits & (0x01 << 14):  # MODE1 (col=5, bit=14)
            flag |= BIT4_MODE1
        if bits & (0x01 << 0):   # ID2   (col=5, bit=0)
            flag |= BIT8_ID2
        if bits & (0x01 << 8):   # ID3   (col=5, bit8)
            flag |= BIT9_ID3
        # ID=3かつID2とID3が0でない場合、未サポート機種
        if (flag & 0x0003) == 3 and (flag & (BIT8_ID2 | BIT9_ID3)) != 0:
            raise Exception("Unsupported Hardware ID")
        return flag

    # [Internal] write extend port for large robo
    def write_extend_ports_for_jog(self,value):
        valstr = "%d" % value
        fp = open(FNAME_IOSTAT,'w')
        fp.write(valstr)
        self.toggle_write_extend_port = 1 - self.toggle_write_extend_port
        if self.toggle_write_extend_port == 1:
            fp.write("\n\n\n") # file size must be changed when every update
        fp.close()

    # [Internal] write extend port for large robo
    def write_mode_io(self, auto_mode):
        idx = 4
        if auto_mode == True:
            value = 0x00000003
        else:
            value = 0x00000000
        return self.write_memory_io_word(idx,value,0xFFFFFFFC)

    # [Internal] read word state (32bits) from whole memory io
    def read_memory_io_word_rb(self, word_addr):
        ret = self.__rblib.ioctrl(word_addr, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF)
        if ret[0] == False:
            self.__sts.set_system_status(self.__sts.SS_NERR, 6)
        return ret[1]

    # [Internal] read word state (32bits) from memory io by shm
    def read_memory_io_word(self, word_addr):
        phy_adr = word_addr*4 + 0x0100
        if phy_adr >= 0x010C and phy_adr < 0x0300:
            ret = shm_read(phy_adr,4).split(",")
            data0 = int(ret[0])
            data1 = int(ret[1])
            data2 = int(ret[2])
            data3 = int(ret[3])
            ret = (data3 << 24) + (data2 << 16) + (data1 << 8) + data0
        else:
            ret = shm_read(phy_adr,1)
        #print "read_memory_io(%d:%08x)=%08X" % (word_addr,word_addr*4 + 0x0100,int(ret))
        return int(ret)

    # [Internal] write word state with mask to whole memory io
    def write_memory_io_word(self, word_addr, data, mask):
        ret = self.__rblib.ioctrl(word_addr, data, mask, 0xFFFFFFFF, 0xFFFFFFFF)
        if ret[0] == False:
            self.__sts.set_system_status(self.__sts.SS_NERR, 6)
        return ret[0]

    # [Internal] read bit state (0 or 1) from whole memory io
    def read_memory_io_bit(self, port_no):
        addr = port_no / 32
        bit = port_no % 32
        data = self.read_memory_io_word(addr)
        if (data & (1<<bit))!=0:
            return 1
        return 0

    # [Internal] write bit state (0 or 1) to whole memory io
    def write_memory_io_bit(self, port_no, st):
        addr = port_no / 32
        bit = port_no % 32
        data = 0
        mask = 0xFFFFFFFF ^ (1<<bit)
        if st==1:
            data |= 1 << bit
        return self.write_memory_io_word(addr,data,mask)

    ## [Internal] Set app status to run in memory I/O
    def set_app_status_to_run(self):
        write_status = 0x00000000  # 0=run, code=00, io_disable=0
        write_mask   = 0xE000FFFF
        self.write_memory_io_word(130, write_status, write_mask)


"""======================================================================
[内部クラス]
  _SysStatus() : システム状態管理
  制御権, コマンドバッファ, プロセス起動状態の管理
======================================================================"""
class _SysStatus(object):
    # PortG1 [ 0:svon,      1:emo,      2:hw_error,     3:abs_lost  ] In/Out
    # PortG2 [ 0:running,   1:sw_error, 2:in_pause,     3:error     ] Out
    # PortG3 [ 0:run,       1:stop,     2:err_reset,    3:pause     ] In
    G1_PORT_NUM    = 4
    G2_PORT_NUM    = 4
    G3_PORT_NUM    = 4
    MAX_ASSIGN_PORT_NO = 0x2FFF

    # Port ID
    G1_SVON        = 0
    G1_EMO         = 1
    G1_HW_ERROR    = 2
    G1_ABS_LOST    = 3
    G2_RUNNING     = 0
    G2_SW_ERROR    = 1
    G2_IN_PAUSE    = 2
    G2_ERROR       = 3
    G3_RUN         = 0
    G3_STOP        = 1
    G3_ERR_RESET   = 2
    G3_PAUSE       = 3

    # command from robsys
    CMD_RBSYS_NONE         = 0
    CMD_RBSYS_RUN          = 1
    CMD_RBSYS_STOP         = 2
    CMD_RBSYS_RESET        = 3
    CMD_RBSYS_PAUSE        = 4
    CMD_RBSYS_REGISTER     = 5

    # App Info
    APP_RUNNING = 0                # not error(0-7)
    APP_EXIT_NORMAL = 1
    APP_EXIT_ERROR = 2
    APP_EXIT_EXCEPTION = 3
    APP_PAUSE = 4
    APP_EXIT_NORMAL_ERROR = 8    # Error (8-15)
    APP_EXIT_CRITICAL_ERROR = 9
    APP_EXIT_USER_NORMAL_ERROR = 10
    APP_EXIT_USER_CRITICAL_ERROR = 11

    # Status value
    SS_BOOT        = 0
    SS_INIT        = 1
    SS_READY       = 2
    SS_INC         = 3
    SS_TEACH       = 4
    SS_JOG         = 5
    SS_RUN         = 6
    SS_PAUSE       = 7
    SS_NERR        = 10
    SS_CERR        = 11
    SS_UNERR       = 12
    SS_UCERR       = 13
    SS_TAG = ("BOOT","INIT","READY","INC","TEACH","JOG","RUN","PAUSE",
                    "","","NERR","CERR","UNERR","UCERR")
    TAG_JOINT = "JNT"
    TAG_EVENT = "EVT"
    TAG_SLAVE = "SLV"
    SS_7seg = ("---","INI","RDY","INC","TCH","JOG","RUN","PAU","","",
                    "E%02d","C%02d","U%02d","R%02d")
    SYSTEM_LOG_INTERVAL = (60*1)    # 1 minutes
    #SYSTEM_LOG_INTERVAL = (10)    # 10 sec (debug)

    # Pause status value
    PS_NONE      = 0  # 000:not pause
    PS_RCV_PAUSE = 2  # 010:recv pause command
    PS_RCV_CONT  = 4  # 100:recv continue command
    PS_RCV_ERROR = 6  # 110:recv error
    PS_RSV       = 1  # 001:(reserved)
    PS_IN_MOTION = 3  # 011:paused in native
    PS_IN_HOOK   = 5  # 101:paused in hook
    PS_IN_BOTH   = 7  # 111:paused in both native and hook

    ## Request for app
    REQ_NONE       = 0x00
    REQ_STOP       = 0x01
    REQ_PAUSE      = 0x02
    REQ_CONTINUE   = 0x04
    REQ_KILL       = 0x08

    FONT7SEG = {'0':0x40, # , # 0
                '1':0x79, # , # 1
                '2':0x24, # , # 2
                '3':0x30, # , # 3
                '4':0x19, # , # 4
                '5':0x12, # , # 5
                '6':0x02, # , # 6
                '7':0x78, # , # 7
                '8':0x00, # , # 8
                '9':0x10, # , # 9
                'A':0x08, # , # A
                'B':0x03, # , # B
                'C':0x27, # , # C
                'D':0x21, # , # D
                'E':0x06, # , # E
                'F':0x0e, # , # F
                'G':0x42, # , # G
                'H':0x0b, # , # H
                'I':0x7b, # , # I
                'J':0x61, # , # J
                'K':0x0a, # , # K
                'L':0x47, # , # L
                'M':0x48, # , # M
                'N':0x2b, # , # N
                'O':0x23, # , # O
                'P':0x0c, # , # P
                'Q':0x18, # , # Q
                'R':0x2f, # , # R
                'S':0x12, # , # S
                'T':0x07, # , # T
                'U':0x63, # , # U
                'V':0x41, # , # V
                'W':0x01, # , # W
                'X':0x09, # , # X
                'Y':0x11, # , # Y
                'Z':0x64, # , # Z
                '-':0x3f} # , # -

    def __init__(self):
        # user program
        self.__prog_name = ''
        self.__prog_name_default = ''
        self.__prog = None
        self._prog_pid = 0
        self.__prog_is_child = False
        self._prog_fout = None
        self._prog_ferr = None
        self.teach_pid = 0
        self._rblib = None

        # command from robsys
        self.command_from_robsys = self.CMD_RBSYS_NONE

        # Assign port number
        self.id_assign_in_G3  = [-1] * self.G3_PORT_NUM
        self.id_assign_out_G1 = [-1] * self.G1_PORT_NUM
        self.id_assign_out_G2 = [-1] * self.G2_PORT_NUM

        # update sysinfo in shm
        sysinfo_str = self.id_assign_in_G3 + [self.id_assign_out_G2[0], self.id_assign_out_G1[0],
            self.id_assign_out_G1[1],  self.id_assign_out_G1[2], self.id_assign_out_G2[1],
            self.id_assign_out_G1[3], self.id_assign_out_G2[2], self.id_assign_out_G2[3]]
        shm_system_write(SYSINFO_ADDR2, sysinfo_str )

        # Port status
        self.st_preset_in_G1  = [0] * self.G1_PORT_NUM
        self.st_preset_in_G3  = [0] * self.G3_PORT_NUM
        self.st_assign_in_G3  = [0] * self.G3_PORT_NUM
        self.st_out_G2        = [0] * self.G2_PORT_NUM

        # Port Pulse input detector
        self.detect_preset_in_G1       = [0] * self.G1_PORT_NUM
        self.detect_preset_in_G3       = [0] * self.G3_PORT_NUM
        self.detect_assign_in_G3       = [0] * self.G3_PORT_NUM
        for i in range(self.G3_PORT_NUM):
            self.detect_preset_in_G3[i] = i611_common._pulse_up()
            self.detect_assign_in_G3[i] = i611_common._pulse_up()

        # App Info
        self.app_exit_code = 0              # error (1～99)
        self.app_run_status = self.APP_EXIT_NORMAL
        self.disable_io_input = 0
        self.app_pause_status = self.PS_NONE

        # for ndstatus log
        self.__logsockname = "/tmp/ndstatus"
        #pylint: disable=maybe-no-member
        self.__logsock = socket.socket(socket.AF_UNIX, socket.SOCK_DGRAM)
        self.last_status_log_time = 0
        self.last_emo = False
        self.is_simulator = False

        # lock
        self.__lock_socket = threading.Lock()
        self.__lock_set_system_status = threading.Lock()

        ## System Status
        self.system_status    = self.SS_INIT
        self.cur_error_no     = 0
        # additional status
        self.now_pause = 0
        self.hardware_id = 0
        self.exists_auto_ready = False

        # Zeus Teaching flag                                # CT210121
        self.is_ZeusTeach = False      		                # CT210121

        ## Request for app
        self.request_to_app = self.REQ_NONE
        self.indicator_data = 0xFFFFFFFF

        # reload init.py
        self._flg_reinit = threading.Event()

    ## [Internal] generate string to 7seg data
    def generate_indicator_data(self, status, code):
        str7seg = ""
        if status<10:
            str7seg = self.SS_7seg[status]
        else:
            if status == 11 and code == 98: # ACPF
                str7seg = "POF"
            else:
                str7seg = self.SS_7seg[status] % code
        print2("[%s]" % str7seg)
        data1 = self.FONT7SEG[str7seg[0]]
        data2 = self.FONT7SEG[str7seg[1]]
        data3 = self.FONT7SEG[str7seg[2]]
        out_data = 0x0020 if status != self.SS_RUN and status != self.SS_PAUSE else 0x00
        out_data |= (data1 & 0x07F) << 8
        out_data |= (data2 & 0x07F) << 16
        out_data |= (data3 & 0x07F) << 24
        out_data |= 0x80808000
        return out_data

    def prog_wait(self):
        if self.__prog is not None and self.__prog_is_child:
            self.__prog.wait()
            if self._prog_fout is not None:
                os.close(self._prog_fout)
                self._prog_fout = None
            if self._prog_ferr is not None:
                os.close(self._prog_ferr)
                self._prog_ferr = None
        else:
            print "warning! user program is zonbie but not child..."

    ## [Internal] Set System Status with error code
    def set_system_status(self, st, err=0, force=False):
        with self.__lock_set_system_status:
            tag = self.SS_TAG[st]
            updated = False
            if force: # not update
                st = self.system_status
                err = self.cur_error_no
            if st != self.system_status or force == True:
                stsmsg = ""
                updated = True
                if st == self.SS_READY or st == self.SS_INC:
                    stsmsg = "%d" % self.st_preset_in_G1[self.G1_EMO]
                elif st == self.SS_JOG:
                    stsmsg = "0" # always 0 during teach
                elif st == self.SS_TEACH:
                    stsmsg = "0" # always 0 during teach
                elif st == self.SS_RUN:
                    stsmsg = "%s" % self.__prog_name
                elif st == self.SS_PAUSE:
                    stsmsg = "%d" % self.app_pause_status
                elif (st == self.SS_NERR or st == self.SS_CERR or st == self.SS_UNERR
                    or st == self.SS_UCERR):
                    if st == self.SS_CERR and err == 98:
                        tag = "POFF"
                        stsmsg = "%d" % self.st_preset_in_G1[self.G1_EMO]
                    else:
                        print "detect error(%d)" % err
                        if err==0:
                            err=99 # unknown
                        stsmsg = "%d,%s" % (err, i611Robot._get_syserr_msg(st,err))
                        self.collect_error_log(st,err)
                self.system_status = st
                self.cur_error_no = err
                self.status_log(tag,stsmsg)
                self.indicator_data = self.generate_indicator_data(st,err)
                self.last_status_log_time = time.time()
                self.last_emo = self.st_preset_in_G1[self.G1_EMO]
            if updated and not force: # output current position
                jnt = self._rblib.jmark()[1:]
                stsmsg = "%.3f,%.3f,%.3f,%.3f,%.3f,%.3f" % (
                    jnt[0],jnt[1],jnt[2],jnt[3],jnt[4],jnt[5])
                self.status_log(self.TAG_JOINT,stsmsg)

    # [Internal] update status log when status is not chage for a while
    def update_status_log(self):
        do_update = False
        now = time.time()
        if (self.system_status < 10 and self.last_status_log_time != 0 and
            now - self.last_status_log_time > self.SYSTEM_LOG_INTERVAL):
            do_update = True
        if (self.system_status < 10 and self.system_status != self.SS_TEACH and
            self.last_emo != self.st_preset_in_G1[self.G1_EMO]):
            self.last_emo = self.st_preset_in_G1[self.G1_EMO]
            do_update = True
        if do_update:
            self.set_system_status(self.system_status,force=True)
            #print "update status log!"

    # [Internal] output ndstatus log
    def status_log(self, tag_id, logmsg):
        with self.__lock_socket:
            now = datetime.datetime.today()
            timestr = "%04d-%02d-%02d %02d:%02d:%02d.%03d" % (now.year,
                now.month, now.day, now.hour, now.minute, now.second, now.microsecond/1000)
            outstr = "%s,%s,%s\n" % (timestr,tag_id,logmsg)
            try:
                #print "sendto=%s" % outstr
                self.__logsock.sendto(outstr, self.__logsockname)
            except Exception:
                print "Cannot write ndstatus log."

    # [Internal] output ndstatus warning log
    def warning_log(self, logmsg):
        tag_id = "WARN"
        with self.__lock_socket:
            now = datetime.datetime.today()
            timestr = "%04d-%02d-%02d %02d:%02d:%02d.%03d" % (now.year,
                now.month, now.day, now.hour, now.minute, now.second, now.microsecond/1000)
            outstr = "%s,%s,%s\n" % (timestr,tag_id,logmsg)
            try:
                #print "sendto=%s" % outstr
                self.__logsock.sendto(outstr, self.__logsockname)
            except Exception:
                print "Cannot write ndstatus warn log."

    # [Internal] utility for ndstatus info
    @classmethod
    def get_serial_no(cls):
        cmd = "/opt/i611/tools/show_serial.sh"
        return subprocess.check_output( cmd ).rstrip()


    # [Internal] utility for ndstatus info
    @classmethod
    def get_romver(cls):
        f = open("/opt/i611/version")
        romver = f.readline()
        f.close()
        f = open("/opt/i611/build")
        build = f.readline()
        f.close()
        return "%s,%s" % (romver.rstrip(), build.rstrip())

    @classmethod
    def get_kernel_ver(cls):
        cmd = "/bin/uname -v"
        return subprocess.check_output( cmd, shell=True ).rstrip()

    # [RemoteAPI] rbsys.set_robtask
    def handle_set_robtask(self, fname):
        if self.st_out_G2[self.G2_RUNNING]==1:
            return [False]
        if os.path.isfile(fname) == False:
            self.__prog_name_default = ''
            ret = [False]
        else:
            self.__prog_name_default = fname
            ret = [True]
        # update sysinfo in shm
        shm_system_write(SYSINFO_ADDR1, (self.__prog_name_default,
            self.__prog_name, self._prog_pid))
        return ret


    # [RemoteAPI] rbsys.get_robtask
    def handle_get_robtask(self):
        if self.st_out_G2[self.G2_RUNNING]==1:
            return [True,  self._prog_pid, self.__prog_name]
        else:
            return [False, self._prog_pid, self.__prog_name_default]


    # [RemoteAPI] rbsys.clear_robtask
    def handle_clear_robtask(self):
        self.__prog_name_default = ''
        # update sysinfo in shm
        shm_system_write(SYSINFO_ADDR1, (self.__prog_name_default,
            self.__prog_name, self._prog_pid))
        return [True]


    # [RemoteAPI] rbsys.req_mcmd
    def handle_req_mcmd(self):
        res = (self.st_out_G2[self.G2_RUNNING],         # 0  status - running
               self.st_preset_in_G1[self.G1_SVON],      # 1  status - svon
               self.st_preset_in_G1[self.G1_EMO],       # 2  status - emo
               self.st_preset_in_G1[self.G1_HW_ERROR],  # 3  status - hw_error
               self.st_out_G2[self.G2_SW_ERROR],        # 4  status - sw_error
               self.st_preset_in_G1[self.G1_ABS_LOST],  # 5  status - abs_lost
               self.st_out_G2[self.G2_IN_PAUSE],        # 6  status - in_pause
               self.st_out_G2[self.G2_ERROR],           # 7  status - error
               self.request_to_app)                     # 8  request for app (bit mask)
        self.request_to_app = self.REQ_NONE
        return res

    # [Internal] clear user prog info
    def clear_rbfname(self):
        print "clearn rbfname"
        self.__prog_name = ''
        self._prog_pid = 0
        self.__prog = None
        self.__prog_is_child = False    # flag must be cleared!
        self._prog_fout = None
        self._prog_ferr = None
        # update sysinfo in shm
        shm_system_write(SYSINFO_ADDR1, (self.__prog_name_default,
            self.__prog_name, self._prog_pid))

    # [RemoteAPI] rbsys.cmd_run
    def handle_cmd_run(self, fname):
        if self.command_from_robsys != self.CMD_RBSYS_NONE:
            print "handle_cmd_run...busy"
            return [False]
        self.__prog_name = fname
        self.command_from_robsys = self.CMD_RBSYS_RUN
        print "handle_cmd_run...exec (%s)" % fname
        return [True]

    # [RemoteAPI] rbsys.cmd_stop
    def handle_cmd_stop(self):
        if self.command_from_robsys != self.CMD_RBSYS_NONE:
            print "handle_cmd_stop...busy"
            return [False]
        self.command_from_robsys = self.CMD_RBSYS_STOP
        print "handle_cmd_stop...exec"
        return [True]

    # [RemoteAPI] rbsys.cmd_reset
    def handle_cmd_reset(self):
        if self.command_from_robsys != self.CMD_RBSYS_NONE:
            print "handle_cmd_reset...busy"
            return [False]
        self.command_from_robsys = self.CMD_RBSYS_RESET
        print "handle_cmd_reset...exec"
        return [True]

    # [RemoteAPI] rbsys.cmd_pause
    def handle_cmd_pause(self):
        if self.command_from_robsys != self.CMD_RBSYS_NONE:
            print "handle_cmd_pause...busy"
            return [False]
        self.command_from_robsys = self.CMD_RBSYS_PAUSE
        print "handle_cmd_pause...exec"
        return [True]


    # [RemoteAPI] rbsys.assign_din
    # Order G3[ 0:run,       1:stop,     2:err_reset,    3:pause     ]
    def handle_assign_din(self, param):

        #割付重複チェック
        if len([k for k,v in Counter(param).items() if (k!=-1) and (v>1)]) > 0:
            return [False]
        # Inhibit assign during user program running, don't care servo on
        if self.st_out_G2[self.G2_RUNNING] == 1:
            print "Error) running=%d, svon=%d" % (self.st_out_G2[self.G2_RUNNING],
                self.st_preset_in_G1[self.G1_SVON])
            return [False]

        for i in range(self.G3_PORT_NUM):
            if i611_common._chkparam(param[i], p_type=int, min=-1,
                max=self.MAX_ASSIGN_PORT_NO)[0]:
#                print "OK)handle assign din=%d,%d" % (i,param[i])

                self.id_assign_in_G3[i] = param[i]
            else:

                self.id_assign_in_G3[i] = -1
                print "Error)handle assign din=%d,%d" % (i,param[i])
                return [False]
        # update sysinfo in shm
        sysinfo_str = self.id_assign_in_G3 + [self.id_assign_out_G2[0], self.id_assign_out_G1[0],
            self.id_assign_out_G1[1],  self.id_assign_out_G1[2], self.id_assign_out_G2[1],
            self.id_assign_out_G1[3], self.id_assign_out_G2[2], self.id_assign_out_G2[3]]
        shm_system_write(SYSINFO_ADDR2, sysinfo_str )

        return [True]


    # [RemoteAPI] rbsys.assign_dout
    # Order [ 0:svon,      1:emo,      2:hw_error,     3:abs_lost  ]
    #       [ 0:running,   1:sw_error, 2:in_pause,     3:error     ]
    def handle_assign_dout(self, param):

        #割付重複チェック
        if len([k for k,v in Counter(param).items() if (k!=-1) and (v>1)]) > 0:
            return [False]

        # Inhibit assign during user program running, don't care servo on
        if self.st_out_G2[self.G2_RUNNING] == 1:
            return [False]

        for i in range(self.G1_PORT_NUM):
            if i611_common._chkparam(param[i], p_type=int, min=-1,
                max=self.MAX_ASSIGN_PORT_NO)[0]:
                self.id_assign_out_G1[i] = param[i]
            else:
                self.id_assign_out_G1[i] = -1
                return [False]
        for i in range(self.G2_PORT_NUM):
            if i611_common._chkparam(param[i+self.G1_PORT_NUM], p_type=int,
                min=-1, max=self.MAX_ASSIGN_PORT_NO)[0]:
                self.id_assign_out_G2[i] = param[i+self.G1_PORT_NUM]
            else:
                self.id_assign_out_G2[i] = -1
                return [False]
        # update sysinfo in shm
        sysinfo_str = self.id_assign_in_G3 + [self.id_assign_out_G2[0], self.id_assign_out_G1[0],
            self.id_assign_out_G1[1],  self.id_assign_out_G1[2], self.id_assign_out_G2[1],
            self.id_assign_out_G1[3], self.id_assign_out_G2[2], self.id_assign_out_G2[3]]
        shm_system_write(SYSINFO_ADDR2, sysinfo_str )
        return [True]


    # [RemoteAPI] rbsys.cmd_register_program
    def handle_cmd_register_program(self, pid):
        if pid == self._prog_pid:
            print "Detect register from child"
            self.__prog_is_child = True
            return [True]   # is child
        else:
            procname = "/proc/%d/cmdline" % pid
            f = None
            try:
                f = open(procname, "r")
                fname = f.readline().strip()
            except IOError:
                traceback.print_exc()
                print "Detect register but no proc info."
                if f is not None:
                    f.close()
                return [False]
            else:
                fname = re.sub(r"^python","",fname)
                fname = re.sub(r".*/","",fname)
                fname = re.sub(r"\0","",fname)
                fname = re.sub(r"-mpdb","",fname)
                f.close()

            if self.system_status == self.SS_READY:
                self._prog_pid = pid
                self.__prog_name = fname
                self.__prog_is_child = False
                self._prog_fout = None
                self._prog_ferr = None
                print "Detect register from others(%d:%s)" % (pid, self.__prog_name)
                if self.command_from_robsys != self.CMD_RBSYS_NONE:
                    print "WARNING! robsys command is busy, but overwritten"
                self.command_from_robsys = self.CMD_RBSYS_REGISTER
                # update sysinfo in shm
                shm_system_write(SYSINFO_ADDR1, (self.__prog_name_default,
                    self.__prog_name, self._prog_pid))
                return [True]

            elif self.system_status == self.SS_INC:
                print "Detect register from others, BUT cancel with inc (%d:%s)" % (pid, fname)
                self.set_system_status(self.SS_NERR, 7)
                return [False]

            else: # launch cancel
                print "Detect register from others, BUT cancel (%d:%s) with invalid state(%d)" % (
                    pid, fname, self.system_status)
                return [False]

    # [RemoteAPI] rbsys.cmd_register_program
    def handle_cmd_debug(self, cmd):
        cmd = int(cmd)
        if cmd == 1:    #
            pass
        elif cmd == 2:  # call init.py again
            if self.system_status != self.SS_READY:
                return [False]
            self._flg_reinit.set()
        elif cmd == 3:  # enc_reset begin
            self.status_log(self.TAG_EVENT,"Enc Reset Begin")
        elif cmd == 4:  # enc_reset complete
            self.status_log(self.TAG_EVENT,"Enc Reset Complete")
        return [True]

    # [RemoteAPI] rbsys._internal_call
    def handle_cmd_internal_call(self, cmd):
#        print "handle_cmd_internal_call ",cmd
        if len(cmd) != 2:
            print "cmd length error"
            return [False]
        if self.system_status == self.SS_RUN or self.system_status == self.SS_PAUSE:
            print "status error",self.system_status
            return [False]
        cmdnum = int(cmd[0])
        opt = ""
        result = ""
        opt = cmd[1]
        if cmdnum == 1:    # set time
            cmdline = "/opt/i611/tools/set_time.py"
        elif cmdnum == 2:  # dump_slaves.py
            cmdline = "/opt/i611/tools/dump_slaves.py"
            result  = "/tmp/cmd_result"
        elif cmdnum == 3:  # show_slavestate.py
            cmdline = "/opt/i611/tools/show_slavestate.py"
            result  = "/tmp/cmd_result"
        else:
            print "unknown cmd error",cmdnum
            return [False]
        fout = None
        if result != "":
            fout = open(result,'w')
            prog = subprocess.Popen(('python', cmdline,opt),stdout=fout)
        else: # no output
            prog = subprocess.Popen(('python', cmdline,opt))
#        print "call prog...",cmdline
        ret = prog.wait()
        if fout is not None:
            fout.close()
        if ret != 0:
            print "cmd failure",ret
            return [False]
        else:
#            print "cmd success",ret
            return [True]

    # [Internal] backup userprog_out.log
    def backup_userprog(self):
        PROGNAME = '/opt/i611/log/userprog_out%s.log'
        #erase oldest file
        curfile = PROGNAME % str(9)
        if os.path.exists(curfile):
            os.remove(curfile)
        # shift files
        for i in range(8,-1,-1):
            curfile = PROGNAME % str(i)
            if os.path.exists(curfile):
                newfile = PROGNAME % str(i+1)
                os.rename(curfile,newfile)
        # rename newest file
        curfile = PROGNAME % ""
        if os.path.exists(curfile):
            newfile = PROGNAME % str(0)
            os.rename(curfile,newfile)

    # [Internal] rbsys.cmd_run
    def exec_cmd_run(self):
        try:
            print "exec_run (f=%s,default=%s)" % (self.__prog_name ,self.__prog_name_default)
            if self.__prog_name == '' or self.__prog_name == None:
                self.__prog_name = self.__prog_name_default
            if not self.exists_auto_ready:
                return 5            # NERR(5)
            if self.__prog_name == '' or self.__prog_name is None:
                return 4            # NERR(4)
            else:
                if not os.path.isfile(self.__prog_name):
                    return 3        # NERR(3)
                self.set_system_status(self.SS_RUN)
                self.backup_userprog()
                FILE_OUT_LOG = '/opt/i611/log/userprog_out.log'
                FILE_ERR_LOG = '/opt/i611/log/userprog_err.log'
                if os.path.exists(FILE_ERR_LOG):
                    if os.path.getsize(FILE_ERR_LOG) > 10*1024*1024: # 10MB
                        os.remove(FILE_ERR_LOG)
                fout = open(FILE_OUT_LOG,'w')
                ferr = open(FILE_ERR_LOG,'a')
                now = datetime.datetime.today()
                timestr = "\nBegin:%04d%02d%02d %02d:%02d:%02d.%03d\n" % (now.year,
                    now.month, now.day, now.hour, now.minute, now.second, now.microsecond/1000)
                fout.write(timestr)
                ferr.write(timestr)
                fout.close()
                ferr.close()
                fout = os.open(FILE_OUT_LOG,os.O_WRONLY|os.O_APPEND)
                ferr = os.open(FILE_ERR_LOG,os.O_WRONLY|os.O_APPEND)
                print2( "fout:%d" % fout)
                print "run process: %s" % self.__prog_name
                self.__prog = subprocess.Popen(('python', self.__prog_name),stdout=fout,stderr=ferr)
                if self.__prog.poll() != None:
                    os.close(fout)
                    os.close(ferr)
                    print "run error"
                    return 3        # NERR(3)
                self._prog_pid = self.__prog.pid
                self.__prog_is_child = True
                self._prog_fout = fout
                self._prog_ferr = ferr
                print "pid=%d" % self._prog_pid
                # update sysinfo in shm
                shm_system_write(SYSINFO_ADDR1, (self.__prog_name_default,
                    self.__prog_name, self._prog_pid))
        except Exception:
            return 3                # NERR(3)
        return 0

    # [Internal] polling user process
    # RET = -1:dead, 0:zonbie, 1:alive
    @staticmethod
    def proc_poll(pid):
        # Not refer exit code of child process
        procpath = "/proc/%d/cmdline" % pid
        if not os.path.exists(procpath):
            return -1
        else:
            proc_f = None
            try:
                proc_f = open(procpath)
                line = proc_f.readline()
            except IOError:
                if proc_f is not None:
                    proc_f.close()
                return 0 # unable to read somehow
            else:
                if proc_f is not None:
                    proc_f.close()
                if line == "":
                    print "Zombie!!"
                    return 0
                else:
                    return 1

    # [Internal] rbsys.cmd_stop
    def exec_cmd_stop(self):
        self.request_to_app |= self.REQ_STOP
        return [True]

    # [Internal] rbsys.cmd_pause
    def exec_cmd_pause(self):
        print "exec_cmd_pause"
        self.request_to_app |= self.REQ_PAUSE
        return [True]

    # [Internal] rbsys.cmd_stop
    def exec_cmd_continue(self):
        print "exec_cmd_continue"
        self.request_to_app |= self.REQ_CONTINUE
        return [True]

    def get_slave_robover(self):
        cur_robover = []
        # Judge zero
        is_zero = False
        zero_idx = [1,2,3,4,6,7]
        try:
            cmd = "ethercat upload -p 0 0x100A 0"
            subprocess.check_output( cmd.split(" "), stderr=subprocess.STDOUT  )
        except Exception:
            is_zero = True
            print "zero slave_robover"

        for i in range(0,6):
            if is_zero:
                i = zero_idx[i]
            cmd = "ethercat upload -p %d 0x100A 0" % i
            ret  =  subprocess.check_output( cmd.split(" ") )
            cur_robover.append(ret.rstrip())
        return cur_robover

    def get_slave_framver(self):
        cur_framver = []
        # Judge zero
        is_zero = False
        zero_idx = [1,2,3,4,6,7]
        try:
            cmd = "ethercat upload -p 0 0x100A 0"
            subprocess.check_output( cmd.split(" "), stderr=subprocess.STDOUT  )
        except Exception:
            is_zero = True
            print "zero slave_framver"

        for i in range(0,6):
            if is_zero:
                i = zero_idx[i]
            cmd = "ethercat upload -p %d 0x5100 0" % i
            ret  =  subprocess.check_output( cmd.split(" ") )
            cur_framver.append(ret.split(" ")[0])
        return cur_framver

    def output_slave_info(self):
        try:
            robover = self.get_slave_robover()
            framver = self.get_slave_framver()
        except subprocess.CalledProcessError:
            stmsg = "Unable to use ethercat"
        else:
            stmsg = ""
            for i in range(6):
                stmsg += "%s,%s," % (robover[i],framver[i])
            stmsg.rstrip(',')
        self.status_log(self.TAG_SLAVE,stmsg)

    def collect_error_log(self,sts=-1,err=-1):
        FNAME_ERRLOG="/opt/i611/tools/arc_err.sh"
        if sts == -1:
            cmd = ('bash', FNAME_ERRLOG)
        else:
            cmd = ('bash', FNAME_ERRLOG, str(sts), str(err))
        p = subprocess.Popen(cmd)
        p.wait()
        result = p.poll()
        print "collect error log(%d)" % result
        if result == 0:
            self.status_log(self.TAG_EVENT,"Error log collection success")
        elif result == 1:
            self.status_log(self.TAG_EVENT,"Ignore error log collection")
        else:
            self.status_log(self.TAG_EVENT,"Error log collection failure")

def main():

    # Create system data
    System = _SysStatus()

    serial_no = System.get_serial_no()
    rom_ver = System.get_romver()
    kernel_ver = System.get_kernel_ver()

    # Start Main Loop
    WatchIO = _watchIO('127.0.0.1', 12345, System)

    # Check hardware info (ID=0:Normal, 2:Large)
    hw_id = WatchIO.detect_hardware()

    # First ndstatus log
    msg = "%s,%s,%d,%s" % (serial_no, rom_ver,hw_id,kernel_ver)
    System.status_log(System.SS_TAG[System.SS_INIT],msg)

    # Wait until Native is ready
    if WatchIO.wait_until_ready()==False:
        System.set_system_status(System.SS_CERR,3)

    WatchIO.start()

    # Start socket handler
    CommTask = _commTask(System)
    CommTask.connect('127.0.0.1', 4000, 10, 256)

    while True:
        # output log
        System.status_log(System.TAG_EVENT,"Call init.py")
        # Launch init.py
        if os.path.isfile(FNAME_INIT) == False:
            System.set_system_status(System.SS_NERR,1)     # NERR(1)
            print 'Error. Init.py was not found'
        else:
            System.set_system_status(System.SS_READY)
            # remove err log if size is exceeded
            FILE_OUT_LOG = '/opt/i611/log/sys_out.log'
            FILE_ERR_LOG = '/opt/i611/log/sys_err.log'
            if os.path.exists(FILE_ERR_LOG):
                if os.path.getsize(FILE_ERR_LOG) > 10*1024*1024: # 10MB
                    os.remove(FILE_ERR_LOG)
            # create log file
            fout = open(FILE_OUT_LOG,'w')
            ferr = open(FILE_ERR_LOG,'a')
            now = datetime.datetime.today()
            timestr = "\nBegin:%04d%02d%02d %02d:%02d:%02d.%03d\n" % (now.year,
                now.month, now.day, now.hour, now.minute, now.second, now.microsecond/1000)
            fout.write(timestr)
            ferr.write(timestr)
            fout.close()
            ferr.close()
            fout = os.open(FILE_OUT_LOG,os.O_WRONLY|os.O_APPEND)
            ferr = os.open(FILE_ERR_LOG,os.O_WRONLY|os.O_APPEND)
            p = subprocess.Popen(('python', FNAME_INIT),stdout=fout,stderr=ferr)
            p.wait()
            os.close(fout)
            os.close(ferr)
            if p.poll() != 0:
                System.set_system_status(System.SS_NERR,2)     # NERR(2)
                print 'Error init.py has error'

        # wait for reinit event
        System._flg_reinit.wait()
        System._flg_reinit.clear()


    # wait until exit
    CommTask.join()


if __name__ == "__main__":
    main()
#[EOF]
